<?php
// matricular_modulos.php — matricula a un alumno en uno o varios módulos (POST JSON o form-data)
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

try {
    require_once __DIR__ . '/includes/db.php'; // Debe exponer $conn o $pdo (PDO)

    /** @var PDO $db */
    if (isset($conn) && $conn instanceof PDO)      { $db = $conn; }
    elseif (isset($pdo) && $pdo instanceof PDO)    { $db = $pdo; }
    else { throw new RuntimeException('Sin conexión PDO ($conn o $pdo).'); }

    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $db->setAttribute(PDO::ATTR_EMULATE_PREPARES, true);

    // ---------- Entrada ----------
    // Acepta application/json y x-www-form-urlencoded
    $contentType = $_SERVER['CONTENT_TYPE'] ?? $_SERVER['HTTP_CONTENT_TYPE'] ?? '';
    $payload = null;

    if (stripos($contentType, 'application/json') !== false) {
        $raw = file_get_contents('php://input');
        $tmp = json_decode($raw, true);
        if (json_last_error() === JSON_ERROR_NONE) { $payload = $tmp; }
    }
    if ($payload === null) { // fallback a POST normal
        $payload = $_POST;
    }
    if (!$payload || !is_array($payload)) {
        echo json_encode(['ok'=>false,'error'=>'Sin datos recibidos']);
        exit;
    }

    // Extrae y normaliza
    $id_alumno = isset($payload['id_alumno']) ? (int)$payload['id_alumno'] : 0;
    $anno      = isset($payload['anno']) ? trim((string)$payload['anno']) : '';
    $modulos   = $payload['modulos'] ?? ($payload['id_modulo'] ?? []);

    if ($id_alumno <= 0) { echo json_encode(['ok'=>false,'error'=>'id_alumno inválido']); exit; }
    if ($anno === '')    { echo json_encode(['ok'=>false,'error'=>'anno obligatorio (ej: 2025/2026)']); exit; }

    if (!is_array($modulos)) { $modulos = [$modulos]; }
    $modulos = array_values(array_unique(array_map('intval', $modulos)));
    $modulos = array_filter($modulos, fn($m) => $m > 0);
    if (!$modulos) { echo json_encode(['ok'=>false,'error'=>'Lista de módulos vacía']); exit; }

    // Verifica existencia del alumno
    $st = $db->prepare("SELECT COUNT(*) FROM alumnos WHERE id = :id");
    $st->execute([':id'=>$id_alumno]);
    if ((int)$st->fetchColumn() === 0) {
        echo json_encode(['ok'=>false,'error'=>'Alumno no existe']); exit;
    }

    // Verifica existencia de módulos
    $ph = implode(',', array_fill(0, count($modulos), '?'));
    $st = $db->prepare("SELECT id FROM modulos WHERE id IN ($ph)");
    $st->execute($modulos);
    $existentes = array_map('intval', $st->fetchAll(PDO::FETCH_COLUMN, 0));
    $no_encontrados = array_values(array_diff($modulos, $existentes));

    // Inserción (ignora duplicados por UNIQUE id_alumno,id_modulo,anno)
    $sql = "INSERT INTO alumnos_modulos (id_alumno, id_modulo, anno)
            VALUES (:id_alumno, :id_modulo, :anno)
            ON DUPLICATE KEY UPDATE id = id"; // no-op

    $ins = $db->prepare($sql);
    $db->beginTransaction();

    $resumen = ['insertados'=>0, 'duplicados'=>0, 'errores'=>[], 'procesados'=>[]];

    foreach ($existentes as $id_modulo) {
        try {
            $ins->execute([
                ':id_alumno' => $id_alumno,
                ':id_modulo' => $id_modulo,
                ':anno'      => $anno
            ]);
            // Con ON DUPLICATE KEY, rowCount() suele ser 1 (insert) o 2 (no-op update). Tratamos >=2 como duplicado.
            $rc = $ins->rowCount();
            if ($rc === 1) {
                $resumen['insertados']++;
                $resumen['procesados'][] = ['id_modulo'=>$id_modulo, 'status'=>'insertado'];
            } else {
                $resumen['duplicados']++;
                $resumen['procesados'][] = ['id_modulo'=>$id_modulo, 'status'=>'ya_existe'];
            }
        } catch (Throwable $e) {
            $resumen['errores'][] = ['id_modulo'=>$id_modulo, 'error'=>$e->getMessage()];
            $resumen['procesados'][] = ['id_modulo'=>$id_modulo, 'status'=>'error'];
        }
    }

    $db->commit();

    echo json_encode([
        'ok' => true,
        'id_alumno' => $id_alumno,
        'anno' => $anno,
        'resumen' => $resumen,
        'modulos_no_encontrados' => $no_encontrados
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
    if (isset($db) && $db->inTransaction()) { $db->rollBack(); }
    // En producción puedes ocultar 'detalle'
    echo json_encode(['ok'=>false,'error'=>'Error interno','detalle'=>$e->getMessage()]);
}
