<?php
// alumnos_importar.php (versión NRE/EXPEDIENTE)
session_start();
require_once 'includes/db.php'; // Debe definir $conn = new PDO(...)
 require_once 'menu.php';

if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

/* ==== Utilidades ==== */
function normaliza_txt($s) {
    $s = trim((string)$s);
    if ($s === '') return '';
    if (!mb_detect_encoding($s, 'UTF-8', true)) {
        $s = @iconv('ISO-8859-1', 'UTF-8//TRANSLIT', $s);
    }
    $s = mb_strtolower($s, 'UTF-8');
    $s = @iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $s);
    $s = preg_replace('/\s+/', ' ', $s);
    return trim($s);
}
function limpiar_bom($s) { return preg_replace('/^\xEF\xBB\xBF/', '', $s); }
function detectar_delimitador($linea) {
    $c = [","=>0,";"=>0,"\t"=>0]; foreach ($c as $k=>$_) $c[$k]=substr_count($linea,$k);
    arsort($c); $k = array_key_first($c); return $c[$k]>0 ? $k : ",";
}
function parse_fecha($str) {
    $str = trim($str);
    if ($str==='') return null;
    $str = str_replace(['.',' '], ['/'],$str);
    $candidatos = ['d/m/Y','d-m-Y','Y-m-d','d/m/y','d-m-y'];
    foreach ($candidatos as $fmt) {
        $dt = DateTime::createFromFormat($fmt, $str);
        if ($dt && $dt->format($fmt) === $str) return $dt->format('Y-m-d');
    }
    // Intento liberal
    $ts = strtotime($str);
    return $ts ? date('Y-m-d',$ts) : null;
}
function limpia_telefono($s) {
    $s = preg_replace('/[^0-9+]/', '', (string)$s);
    return mb_substr($s, 0, 25);
}

/* ==== Sinónimos de cabeceras -> campo ==== */
$SINONIMOS = [
    'nre'              => ['nre','numero regional de estudiante','nº regional'],
    'expediente'       => ['expediente','num expediente','nº expediente'],
    'nombre'           => ['nombre','name'],
    'apellido1'        => ['apellido 1','apellido1','primer apellido','ap1'],
    'apellido2'        => ['apellido 2','apellido2','segundo apellido','ap2'],
    'dni'              => ['dni','nif','nie','documento','doc'],
    'grupo'            => ['grupo','curso'],
    'subgrupo'         => ['subgrupo','sub-grupo','sub grupo'],
    'fecha_nacimiento' => ['fecha de nacimiento','fecha nacimiento','fecha nac','nacimiento','fecha de na'],
    'telefono'         => ['telefono','teléfono','tlf','movil','móvil','telefono 1','teléfono 1'],
    'telefono2'        => ['telefono 2','teléfono 2','tlf2','movil2','móvil 2'],
    'email_personal'   => ['email personal','correo personal','email','correo'],
    'email_oficial'    => ['email oficial','correo oficial','email institucional','correo institucional'],
    // Por compatibilidad con tu esquema previo:
    'curso'            => ['curso (compat)'] // lo construiremos desde grupo/subgrupo
];

/* Mapear cabeceras a índices */
function mapear_cabeceras($headers, $SINONIMOS) {
    $map = array_fill_keys(array_keys($SINONIMOS), null);
    foreach ($headers as $idx => $h) {
        $hNorm = normaliza_txt($h);
        foreach ($SINONIMOS as $campo => $aliases) {
            foreach ($aliases as $alias) {
                if ($hNorm === normaliza_txt($alias)) {
                    if ($map[$campo] === null) $map[$campo] = $idx;
                }
            }
        }
    }
    // Obligatorios mínimos del CSV
    $ok = ($map['nombre'] !== null) && ($map['apellido1'] !== null || $map['apellido2'] !== null);
    return [$ok, $map];
}

/* ==== Estado ==== */
$errores = [];
$resumen = ['insertados'=>0,'omitidos'=>0,'errores'=>0,'actualizados'=>0];
$logErrores = [];

/* ==== POST ==== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $errores[] = 'Token CSRF inválido. Recarga la página.';
    }

    $modo = $_POST['modo_duplicado'] ?? 'omitir'; // omitir | actualizar | error

    if (!isset($_FILES['csv']) || $_FILES['csv']['error'] !== UPLOAD_ERR_OK) {
        $errores[] = 'No se ha subido el archivo correctamente.';
    } else {
        $tmp = $_FILES['csv']['tmp_name'];
        $ext = strtolower(pathinfo($_FILES['csv']['name'], PATHINFO_EXTENSION));
        if (!in_array($ext, ['csv','txt'])) $errores[] = 'Extensión no permitida. Usa .csv o .txt';
    }

    if (empty($errores)) {
        $fh = fopen($tmp, 'r');
        if (!$fh) $errores[] = 'No se pudo abrir el archivo.';
        else {
            $primera = fgets($fh); rewind($fh);
            $delim = detectar_delimitador(limpiar_bom($primera));

            $csv = new SplFileObject($tmp);
            $csv->setFlags(SplFileObject::READ_CSV | SplFileObject::SKIP_EMPTY);
            $csv->setCsvControl($delim);

            $headers = $csv->fgetcsv();
            if (!$headers) $errores[] = 'No se pudieron leer las cabeceras.';
            else {
                $headers[0] = limpiar_bom($headers[0]);
                [$okMap, $map] = mapear_cabeceras($headers, $SINONIMOS);
                if (!$okMap) $errores[] = 'Faltan columnas obligatorias: NOMBRE y APELLIDO(s).';
            }
        }
    }

    if (empty($errores)) {
        // Consultas preparadas
        $sqlInsert = "INSERT INTO alumnos
            (nre, expediente, nombre, apellidos, apellido1, apellido2, dni, curso, grupo, subgrupo, fecha_nacimiento, telefono, telefono2, email_personal, email_oficial)
            VALUES
            (:nre, :expediente, :nombre, :apellidos, :apellido1, :apellido2, :dni, :curso, :grupo, :subgrupo, :fecha_nacimiento, :telefono, :telefono2, :email_personal, :email_oficial)";
        $stmtInsert = $conn->prepare($sqlInsert);

        // ON DUPLICATE (necesita índices únicos en nre/expediente)
        $sqlUpsert = "INSERT INTO alumnos
            (nre, expediente, nombre, apellidos, apellido1, apellido2, dni, curso, grupo, subgrupo, fecha_nacimiento, telefono, telefono2, email_personal, email_oficial)
            VALUES
            (:nre, :expediente, :nombre, :apellidos, :apellido1, :apellido2, :dni, :curso, :grupo, :subgrupo, :fecha_nacimiento, :telefono, :telefono2, :email_personal, :email_oficial)
            ON DUPLICATE KEY UPDATE
              nombre = VALUES(nombre),
              apellidos = VALUES(apellidos),
              apellido1 = VALUES(apellido1),
              apellido2 = VALUES(apellido2),
              dni = VALUES(dni),
              curso = VALUES(curso),
              grupo = VALUES(grupo),
              subgrupo = VALUES(subgrupo),
              fecha_nacimiento = VALUES(fecha_nacimiento),
              telefono = VALUES(telefono),
              telefono2 = VALUES(telefono2),
              email_personal = VALUES(email_personal),
              email_oficial = VALUES(email_oficial)";
        $stmtUpsert = $conn->prepare($sqlUpsert);

        // Búsquedas para modo omitir/error
        $findByNre = $conn->prepare("SELECT id FROM alumnos WHERE nre = :nre LIMIT 1");
        $findByExp = $conn->prepare("SELECT id FROM alumnos WHERE expediente = :exp LIMIT 1");

        $conn->beginTransaction();
        try {
            $linea = 1;
            foreach ($csv as $row) {
                $linea++;
                if ($row === [null] || $row === false) continue;

                $needCols = count($headers);
                if (count($row) < $needCols) $row = array_pad($row, $needCols, '');

                // Campos del CSV
                $get = function($campo) use ($map, $row) {
                    return ($map[$campo] !== null) ? trim((string)$row[$map[$campo]]) : '';
                };

                $nre        = mb_substr($get('nre'), 0, 30);
                $expediente = mb_substr($get('expediente'), 0, 50);
                $nombre     = mb_substr($get('nombre'), 0, 100);
                $ap1        = mb_substr($get('apellido1'), 0, 100);
                $ap2        = mb_substr($get('apellido2'), 0, 100);
                $dni        = ($get('dni') === '') ? null : mb_substr($get('dni'), 0, 20);
                $grupo      = mb_substr($get('grupo'), 0, 50);
                $subgrupo   = mb_substr($get('subgrupo'), 0, 50);
                $fechaNac   = parse_fecha($get('fecha_nacimiento'));
                $tel1       = $get('telefono');  $tel1 = $tel1 !== '' ? limpia_telefono($tel1) : null;
                $tel2       = $get('telefono2'); $tel2 = $tel2 !== '' ? limpia_telefono($tel2) : null;
                $mailPer    = ($get('email_personal') === '') ? null : mb_substr(trim($get('email_personal')), 0, 150);
                $mailOfi    = ($get('email_oficial')  === '') ? null : mb_substr(trim($get('email_oficial')),  0, 150);

                if ($nombre === '' || ($ap1==='' && $ap2==='')) {
                    $resumen['errores']++;
                    $logErrores[] = "Línea $linea: faltan nombre o apellidos.";
                    continue;
                }

                $apellidos = trim($ap1 . ' ' . $ap2);
                // Compatibilidad: si tienes la columna 'curso' previa, podemos usar 'grupo' (y añadir subgrupo si quieres)
                $curso = $grupo . ($subgrupo ? " - $subgrupo" : "");

                $params = [
                    ':nre'              => ($nre === '') ? null : $nre,
                    ':expediente'       => ($expediente === '') ? null : $expediente,
                    ':nombre'           => $nombre,
                    ':apellidos'        => ($apellidos === '') ? null : $apellidos,
                    ':apellido1'        => ($ap1 === '') ? null : $ap1,
                    ':apellido2'        => ($ap2 === '') ? null : $ap2,
                    ':dni'              => $dni,
                    ':curso'            => ($curso === '') ? null : $curso,
                    ':grupo'            => ($grupo === '') ? null : $grupo,
                    ':subgrupo'         => ($subgrupo === '') ? null : $subgrupo,
                    ':fecha_nacimiento' => $fechaNac,
                    ':telefono'         => $tel1,
                    ':telefono2'        => $tel2,
                    ':email_personal'   => $mailPer,
                    ':email_oficial'    => $mailOfi
                ];

                $tieneClave = ($nre !== '') || ($expediente !== '');

                if ($modo === 'actualizar' && $tieneClave) {
                    $stmtUpsert->execute($params);
                    $resumen['actualizados']++; // cuenta conjunta insert/update
                } else {
                    // omitir o error -> mirar duplicado por NRE o EXPEDIENTE (prioriza NRE)
                    $duplicado = false;
                    if ($nre !== '') { $findByNre->execute([':nre'=>$nre]); $duplicado = (bool)$findByNre->fetchColumn(); }
                    if (!$duplicado && $expediente !== '') { $findByExp->execute([':exp'=>$expediente]); $duplicado = (bool)$findByExp->fetchColumn(); }

                    if ($duplicado) {
                        if ($modo === 'omitir') {
                            $resumen['omitidos']++;
                            continue;
                        } else { // error
                            $resumen['errores']++;
                            $logErrores[] = "Línea $linea: duplicado por " . ($nre!=='' ? "NRE ($nre)" : "EXPEDIENTE ($expediente)");
                            continue;
                        }
                    }
                    $stmtInsert->execute($params);
                    $resumen['insertados']++;
                }
            }
            $conn->commit();
        } catch (Throwable $e) {
            $conn->rollBack();
            $errores[] = 'Error durante la importación. Se ha deshecho la operación.';
        }
    }

    // Rotar CSRF
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<title>Importar alumnos (NRE/EXPEDIENTE)</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
 body{font-family:system-ui,-apple-system,Segoe UI,Roboto,Arial,sans-serif;background:#f7f7f9;margin:0;padding:20px}
 .card{max-width:860px;margin:0 auto;background:#fff;border:1px solid #e5e7eb;border-radius:12px;padding:24px;box-shadow:0 2px 10px rgba(0,0,0,.05)}
 h1{margin-top:0;font-size:22px}
 .row{display:flex;gap:16px;flex-wrap:wrap}
 .col{flex:1;min-width:240px}
 label{display:block;font-weight:600;margin-bottom:6px}
 select,input[type="file"]{width:100%;padding:10px 12px;border:1px solid #d1d5db;border-radius:8px;font-size:14px;background:#fff}
 .msg{margin-bottom:12px;padding:10px 12px;border-radius:8px}
 .msg.error{background:#fee2e2;color:#991b1b;border:1px solid #fecaca}
 .msg.ok{background:#dcfce7;color:#14532d;border:1px solid #bbf7d0}
 ul.errors{margin:0 0 8px 16px}
 .resumen{background:#f3f4f6;border:1px solid #e5e7eb;padding:10px 12px;border-radius:8px}
 .actions{margin-top:16px;display:flex;gap:10px}
 button{border:none;padding:10px 14px;border-radius:8px;cursor:pointer;font-weight:600;background:#2563eb;color:#fff}
 .tip{font-size:12px;color:#6b7280;margin-top:8px}
</style>
</head>
<body>
<div class="card">
    <h1>Importar alumnos desde CSV (incluye NRE/EXPEDIENTE, teléfonos y emails)</h1>

    <?php if (!empty($errores)): ?>
        <div class="msg error">
            <strong>Se han encontrado errores:</strong>
            <ul class="errors">
                <?php foreach ($errores as $e): ?><li><?= htmlspecialchars($e) ?></li><?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <?php if ($_SERVER['REQUEST_METHOD'] === 'POST' && empty($errores)): ?>
        <div class="msg ok">
            <strong>Importación finalizada.</strong>
            <div class="resumen">
                Insertados: <?= (int)$resumen['insertados'] ?><br>
                Actualizados (modo actualizar): <?= (int)$resumen['actualizados'] ?><br>
                Omitidos (duplicados): <?= (int)$resumen['omitidos'] ?><br>
                Errores: <?= (int)$resumen['errores'] ?><br>
            </div>
            <?php if (!empty($logErrores)): ?>
                <p class="tip"><strong>Detalle de errores:</strong></p>
                <ul class="errors">
                    <?php foreach ($logErrores as $le): ?><li><?= htmlspecialchars($le) ?></li><?php endforeach; ?>
                </ul>
            <?php endif; ?>
        </div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data" autocomplete="off">
        <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']) ?>">

        <div class="row">
            <div class="col">
                <label for="csv">Archivo CSV</label>
                <input type="file" id="csv" name="csv" accept=".csv,.txt" required>
                <div class="tip">Se detecta automáticamente el delimitador (, ; o tabulación) y las cabeceras.</div>
            </div>
            <div class="col">
                <label for="modo_duplicado">Si existe (por NRE o EXPEDIENTE)…</label>
                <select id="modo_duplicado" name="modo_duplicado">
                    <option value="omitir">Omitir ese registro</option>
                    <option value="actualizar">Actualizar campos</option>
                    <option value="error">Marcar como error</option>
                </select>
                <div class="tip">Para “Actualizar” crea índices únicos en <code>nre</code> y/o <code>expediente</code>.</div>
            </div>
        </div>

        <div class="actions">
            <button type="submit">Importar</button>
        </div>
    </form>

    <p class="tip">Cabeceras esperadas (o sinónimos): NRE, EXPEDIENTE, APELLIDO 1, APELLIDO 2, NOMBRE, GRUPO, SUBGRUPO, FECHA DE NACIMIENTO, TELÉFONO, TELÉFONO 2, EMAIL PERSONAL, EMAIL OFICIAL. (Admite acentos y variantes).</p>
</div>
</body>
</html>
